<?php

namespace App\Http\Controllers\Dashboard\Admin;

use App\Models\Plan;
use App\Enum\PlanType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Http\Requests\StorePlanRequest;
use App\Http\Requests\UpdatePlanRequest;

class PlanController extends Controller
{
    public function index()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Plans', 'active' => true]
        ];

        $plans = Plan::latest()->get();

        $data = [
            'title' => 'Plans',
            'breadcrumbs' => $breadcrumbs,
            'plans' => $plans,
        ];

        return view('dashboard.admin.plan.index', $data);
    }

    public function create()
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Plans', 'url' => route('admin.plan.index')],
            ['label' => 'Create Plan', 'active' => true]
        ];

        $planTypes = PlanType::cases();

        $data = [
            'title' => 'Create Plan',
            'breadcrumbs' => $breadcrumbs,
            'planTypes' => $planTypes

        ];

        return view('dashboard.admin.plan.create', $data);
    }

    public function store(StorePlanRequest $request)
    {
        $validated = $request->validated();

        try {
            DB::beginTransaction();

            Plan::create($validated);

            DB::commit();

            return redirect()->route('admin.plan.index')
                ->with('success', 'Plan created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }

    public function edit(Plan $plan)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Plans', 'url' => route('admin.plan.index')],
            ['label' => 'Edit Plan', 'active' => true]
        ];

        $planTypes = PlanType::cases();

        $data = [
            'title' => 'Edit Plan',
            'breadcrumbs' => $breadcrumbs,
            'plan' => $plan,
            'planTypes' => $planTypes
        ];

        return view('dashboard.admin.plan.edit', $data);
    }

    public function update(UpdatePlanRequest $request, Plan $plan)
    {
        $validated = $request->validated();

        try {
            DB::beginTransaction();

            $plan->update($validated);

            DB::commit();

            return redirect()->route('admin.plan.index')
                ->with('success', 'Plan updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }

    public function show(Plan $plan)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('admin.dashboard')],
            ['label' => 'Plans', 'url' => route('admin.plan.index')],
            ['label' => 'Plan Details', 'active' => true]
        ];

        $data = [
            'title' => 'Plan Details',
            'breadcrumbs' => $breadcrumbs,
            'plan' => $plan,
        ];

        return view('dashboard.admin.plan.show', $data);
    }

    public function delete(Plan $plan)
    {
        try {
            DB::beginTransaction();

            $plan->delete();

            DB::commit();

            return redirect()->route('admin.plan.index')
                ->with('success', 'Plan deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }
}
