<?php

namespace App\Http\Controllers\Dashboard\User;

use App\Models\User;
use App\Models\Wallet;
use App\Trait\FileUpload;
use App\Enum\WalletStatus;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Enum\TransactionType;
use App\Enum\TransactionMethod;
use App\Enum\TransactionDirection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

class DepositController extends Controller
{
    use FileUpload;

    public function __construct()
    {
        $this->middleware(['registeredUser', 'kycVerification']);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $user = Auth::user();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Deposits', 'active' => true]
        ];

        $transactions = Transaction::where('user_id', $user->id)->where('type', TransactionType::DEPOSIT)->latest()->get();

        $data = [
            'title' => 'Deposits',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'transactions' => $transactions
        ];

        return view('dashboard.user.deposit.index', $data);
    }

    public function method()
    {
        $user = Auth::user();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Deposits', 'url' => route('user.deposit.index')],
            ['label' => 'Deposit Method', 'active' => true],
        ];

        $wallets = Wallet::where('status', WalletStatus::ACTIVE)->get();

        $data = [
            'title' => 'Deposit Method',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'wallets' => $wallets
        ];

        return view('dashboard.user.deposit.method', $data);
    }

    public function payment(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Deposits', 'url' => route('user.deposit.index')],
            ['label' => 'Deposit Method', 'url' => route('user.deposit.method')],
            ['label' => 'Payment', 'active' => true],
        ];

        $wallet = Wallet::where('uuid', $uuid)->firstOrFail();

        $data = [
            'title' => 'Payment',
            'breadcrumbs' => $breadcrumbs,
            'wallet' => $wallet
        ];

        return view('dashboard.user.deposit.payment', $data);
    }
    public function paymentProof(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Deposits', 'url' => route('user.deposit.index')],
            ['label' => 'Deposit Method', 'url' => route('user.deposit.method')],
            ['label' => 'Payment', 'url' => route('user.deposit.payment', $uuid)],
            ['label' => 'Payment Proof', 'active' => true],
        ];

        $wallet = Wallet::where('uuid', $uuid)->firstOrFail();

        $data = [
            'title' => 'Payment Proof',
            'breadcrumbs' => $breadcrumbs,
            'wallet' => $wallet
        ];

        return view('dashboard.user.deposit.payment_proof', $data);
    }


    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request, string $uuid)
    {
        $request->validate([
            'amount' => ['required', 'numeric'],
            'transaction_hash' => ['required', 'string'],
            'proof' => ['required', 'image', 'mimes:jpeg,png,jpg,gif,svg', 'max:2048'],
            'note' => ['required', 'string']
        ]);

        $user = Auth::user();

        $wallet = Wallet::where('uuid', $uuid)->firstOrFail();

        $meta['deposit'] = [
            'transaction_hash' => $request->transaction_hash,
            'wallet_name' => $wallet->name,
            'wallet_symbol' => $wallet->symbol,
            'wallet_address' => $wallet->address,
            'wallet_network' => $wallet->network,
        ];

        try {
            DB::beginTransaction();

            Transaction::create([
                'user_id' => $user->id,
                'type' => TransactionType::DEPOSIT,
                'direction' => TransactionDirection::CREDIT,
                'description' => 'Deposit via ' . $wallet->name,
                'amount' => $request->amount,
                'method' => TransactionMethod::CRYPTO,
                'proof' => $this->updateFile($request, 'proof', uploadPathUser('deposit'), $wallet?->proof),
                'transaction_at' => now(),
                'reference_id' => generateReferenceId(),
                'note' => $request->note,
                'meta' => $meta
            ]);

            DB::commit();

            return redirect()->route('user.deposit.index')->with('success', 'Your payment has been received and is currently being verified. You will be notified once the process is complete.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->route('user.deposit.index')->with('error', config('app.messages.error'));
        }
    }

    public function show(string $uuid)
    {
        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Deposits', 'url' => route('user.deposit.index')],
            ['label' => 'Deposit Method', 'url' => route('user.deposit.method')],
            ['label' => 'Payment', 'url' => route('user.deposit.payment', $uuid)],
            ['label' => 'Payment Proof', 'url' => route('user.deposit.payment.proof', $uuid)],
            ['label' => 'Deposit Details', 'active' => true],
        ];

        $user = Auth::user();

        $transaction = $user->transaction()->where('uuid', $uuid)->firstOrFail();

        $data = [
            'title' => 'Deposit Details',
            'breadcrumbs' => $breadcrumbs,
            'user' => $user,
            'transaction' => $transaction
        ];

        return view('dashboard.user.deposit.show', $data);
    }
}
