<?php

namespace App\Http\Controllers\Dashboard\User;

use App\Models\Plan;
use App\Models\User;
use App\Models\Wallet;
use App\Models\UserPlan;
use App\Trait\FileUpload;
use App\Enum\WalletStatus;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Enum\TransactionType;
use App\Enum\TransactionMethod;
use App\Enum\TransactionDirection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

class PlanController extends Controller
{

    use FileUpload;

    public function __construct()
    {
        $this->middleware(['registeredUser', 'kycVerification']);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $user = Auth::user();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Trading Plans', 'active' => true]
        ];

        $plans = Plan::latest()->get();

        $data = [
            'title' => 'Trading Plans',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'plans' => $plans
        ];

        return view('dashboard.user.plan.index', $data);
    }

    public function paymentMethod(string $uuid)
    {
        $user = Auth::user();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Trading Plans', 'url' => route('user.plan.index')],
            ['label' => 'Payment Method', 'active' => true]
        ];

        $plan = Plan::where('uuid', $uuid)->firstOrFail();
        $wallets = Wallet::where('status', WalletStatus::ACTIVE)->get();

        $data = [
            'title' => 'Payment Method',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'plan' => $plan,
            'wallets' => $wallets
        ];

        return view('dashboard.user.plan.payment_method', $data);
    }

    public function payment(string $planUUID, string $walletUUID)
    {
        $user = Auth::user();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Trading Plans', 'url' => route('user.plan.index')],
            ['label' => 'Payment Method', 'url' => route('user.plan.payment.method', $planUUID)],
            ['label' => 'Payment', 'active' => true]
        ];

        $plan = Plan::where('uuid', $planUUID)->firstOrFail();
        $wallet = Wallet::where('uuid', $walletUUID)->firstOrFail();

        $data = [
            'title' => 'Payment',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'plan' => $plan,
            'wallet' => $wallet
        ];

        return view('dashboard.user.plan.payment', $data);
    }

    public function paymentProof(string $planUUID, string $walletUUID)
    {
        $user = Auth::user();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Trading Plans', 'url' => route('user.plan.index')],
            ['label' => 'Payment Method', 'url' => route('user.plan.payment.method', $planUUID)],
            ['label' => 'Payment', 'url' => route('user.plan.payment', [$planUUID, $walletUUID])],
            ['label' => 'Payment Proof', 'active' => true]
        ];

        $plan = Plan::where('uuid', $planUUID)->firstOrFail();
        $wallet = Wallet::where('uuid', $walletUUID)->firstOrFail();

        $data = [
            'title' => 'Payment Proof',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'plan' => $plan,
            'wallet' => $wallet
        ];

        return view('dashboard.user.plan.payment_proof', $data);
    }

    public function paymentProofStore(Request $request, string $planUUID, string $walletUUID)
    {
        $request->validate([
            'amount' => ['required', 'numeric'],
            'transaction_hash' => ['required', 'string'],
            'proof' => ['required', 'image', 'mimes:jpeg,png,jpg,gif,svg', 'max:2048'],
            'note' => ['required', 'string']
        ]);

        $user = Auth::user();
        $plan = Plan::where('uuid', $planUUID)->firstOrFail();
        $wallet = Wallet::where('uuid', $walletUUID)->firstOrFail();

        $meta['plan'] = [
            'transaction_hash' => $request->transaction_hash,
            'wallet_name' => $wallet->name,
            'wallet_symbol' => $wallet->symbol,
            'wallet_address' => $wallet->address,
            'wallet_network' => $wallet->network,
            'plan_name' => $plan->name,
            'plan_type' => $plan->type,
            'plan_min_amount' => $plan->minimum,
            'plan_max_amount' => $plan->maximum,
            'plan_daily_profit' => $plan->daily_profit,
            'plan_daily_profit_duration' => $plan->daily_profit_duration,
            'plan_trades_per_day' => $plan->trades_per_day,
            'plan_features' => $plan->features,
        ];

        try {
            DB::beginTransaction();

            $userPlan = UserPlan::create([
                'user_id' => $user->id,
                'plan_id' => $plan->id,
                'amount' => $request->amount,
                'daily_profit' => $plan->daily_profit,
                'duration' => $plan->daily_profit_duration,
                'trades_per_day' => $plan->trades_per_day,
            ]);

            Transaction::create([
                'user_id' => $user->id,
                'type' => TransactionType::PLAN_SUBSCRIPTION,
                'direction' => TransactionDirection::DEBIT,
                'description' => 'Plan Subscription',
                'amount' => $request->amount,
                'method' => TransactionMethod::CRYPTO,
                'proof' => $this->updateFile($request, 'proof', uploadPathUser('deposit'), $wallet?->proof),
                'transaction_at' => now(),
                'reference_id' => generateReferenceId(),
                'other_reference_id' => $userPlan->uuid,
                'note' => $request->note,
                'meta' => $meta
            ]);

            DB::commit();

            return redirect()->route('user.plan.index')->with('success', 'Payment proof has been submitted successfully, and is now awaiting administrative approval.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->route('user.plan.index')->with('error', $e->getMessage());
        }
    }
}
