<?php

namespace App\Http\Controllers\Dashboard\User;

use App\Models\Plan;
use App\Models\User;
use App\Enum\PlanType;
use App\Models\Wallet;
use App\Enum\WalletStatus;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Enum\TransactionType;
use App\Enum\TransactionMethod;
use App\Enum\TransactionDirection;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Models\PremiumSignalSubscription;
use App\Trait\FileUpload;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class PremiumSignalSubscriptionController extends Controller
{
    use FileUpload;

    public function __construct()
    {
        $this->middleware(['registeredUser', 'kycVerification']);
    }

    /**
     * Handle the incoming request.
     */
    public function index()
    {
        $user = Auth::user();

        $breadcrumbs = [
            ['label' => config('app.name'), 'url' => '/'],
            ['label' => 'Dashboard', 'url' => route('user.dashboard')],
            ['label' => 'Premium Signal Subscription', 'active' => true],
        ];

        $plans = Plan::where('type', PlanType::PREMIUM)->get();
        $wallets = Wallet::where('status', WalletStatus::ACTIVE)->get();

        $data = [
            'title' => 'Premium Signal Subscription',
            'user' => $user,
            'breadcrumbs' => $breadcrumbs,
            'plans' => $plans,
            'wallets' => $wallets
        ];

        return view('dashboard.user.premium_signal_subscription.index', $data);
    }

    public function store(Request $request)
    {
        $request->validate([
            'plan_id' => 'required|integer',
            'telegram_username' => 'required|string',
            'transaction_id' => 'required|string',
            'proof' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048'
        ]);

        try {
            $user = Auth::user();
            $plan = Plan::where('id', $request->plan_id)->firstOrFail();

            if (PremiumSignalSubscription::where('user_id', $user->id)->where('plan_id', $plan->id)->exists()) {
                return redirect()->route('user.premium.signal.subscription.index')->with('error', 'You already have a premium signal subscription.');
            }

            DB::beginTransaction();

            PremiumSignalSubscription::create([
                'user_id' => $user->id,
                'plan_id' => $request->plan_id,
                'telegram_username' => $request->telegram_username,
                'transaction_id' => $request->transaction_id,
                'proof' => $this->uploadFile($request, 'proof', uploadPathUser('premium_signal_subscription'))
            ]);

            DB::commit();

            return redirect()->route('user.premium.signal.subscription.index')->with('success', 'Premium account created successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage());
            return redirect()->back()->with('error', config('app.messages.error'));
        }
    }
}
