<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use App\Enum\Currency;
use App\Enum\UserRole;
use App\Enum\UserIdType;
use App\Enum\UserStatus;
use App\Enum\UserKycStatus;
use Illuminate\Support\Str;
use App\Enum\UserAccountType;
use App\Enum\UserAccountMode;
use App\Enum\UserTwoFactorEnabled;
use Illuminate\Notifications\Notifiable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $guarded = [];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'last_login_at' => 'datetime',
            'password' => 'hashed',
            'role' => UserRole::class,
            'status' => UserStatus::class,
            'two_factor_enabled' => UserTwoFactorEnabled::class,
            'kyc_status' => UserKycStatus::class,
            'id_type' => UserIdType::class,
            'account_type' => UserAccountType::class,
            'currency' => Currency::class,
            'account_mode' => UserAccountMode::class,
        ];
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (empty($model->uuid)) {
                $model->uuid = (string) Str::uuid();
            }
        });
    }

    public function twoFactorEnabled(): bool
    {
        return $this->two_factor_enabled === UserTwoFactorEnabled::ENABLED;
    }

    public function kycPendingAndHasDocument(): bool
    {
        if ($this->id_front != null && $this->id_back != null && $this->id_type != null && $this->kyc_status->value === UserKycStatus::PENDING->value) {
            return true;
        }

        return false;
    }

    public function kycPendingAndHasNoDocument(): bool
    {
        if ($this->kyc_status->value === UserKycStatus::PENDING->value) {
            return true;
        }

        return false;
    }

    public function kycApproved(): bool
    {
        if ($this->kyc_status->value === UserKycStatus::APPROVED->value) {
            return true;
        }

        return false;
    }

    public function kycRejected(): bool
    {
        if ($this->kyc_status->value === UserKycStatus::REJECTED->value) {
            return true;
        }

        return false;
    }

    public function transaction()
    {
        return $this->hasMany(Transaction::class);
    }

    public function notification()
    {
        return $this->hasMany(Notification::class);
    }

    public function trades()
    {
        return $this->hasMany(Trade::class);
    }

    public function userPlans()
    {
        return $this->hasMany(UserPlan::class);
    }
}
